# --
# Copyright (C) 2001-2021 OTRS AG, https://otrs.com/
# Copyright (C) 2021 Znuny GmbH, https://znuny.org/
# --
# This software comes with ABSOLUTELY NO WARRANTY. For details, see
# the enclosed file COPYING for license information (GPL). If you
# did not receive this file, see https://www.gnu.org/licenses/gpl-3.0.txt.
# --

use strict;
use warnings;
use utf8;

use vars (qw($Self));

my $ConfigObject = $Kernel::OM->Get('Kernel::Config');
my $HelperObject = $Kernel::OM->Get('Kernel::System::UnitTest::Helper');
my $Selenium     = $Kernel::OM->Get('Kernel::System::UnitTest::Selenium');

my $IsITSMIncidentProblemManagementInstalled
    = $Kernel::OM->Get('Kernel::System::Util')->IsITSMIncidentProblemManagementInstalled();

# do not checkmx
$HelperObject->ConfigSettingChange(
    Valid => 1,
    Key   => 'CheckEmailAddresses',
    Value => 0,
);

# this test is to check that when AgentTicketPhone is loaded already with
# customer data on it (like when doing Split), the dropdown of Service is
# prefilled with the correct data. This is because of bug
# http://bugs.otrs.org/show_bug.cgi?id=7060

$Selenium->RunTest(
    sub {

        my $CacheObject   = $Kernel::OM->Get('Kernel::System::Cache');
        my $ArticleObject = $Kernel::OM->Get('Kernel::System::Ticket::Article');

        # update sysconfig settings
        $HelperObject->ConfigSettingChange(
            Valid => 1,
            Key   => 'Ticket::Service',
            Value => 1,
        );

        # create test user and login
        my $TestUserLogin = $HelperObject->TestUserCreate(
            Groups => ['users'],
        ) || die "Did not get test user";

        $Selenium->Login(
            Type     => 'Agent',
            User     => $TestUserLogin,
            Password => $TestUserLogin,
        );

        # create a customer and a ticket from that customer as previous
        # steps to do the selenium testing

        my $ServiceObject = $Kernel::OM->Get('Kernel::System::Service');
        my $TicketObject  = $Kernel::OM->Get('Kernel::System::Ticket');

        # create a test customer
        my $TestUserCustomer = $HelperObject->TestCustomerUserCreate()
            || die "Did not get test customer user";

        # create a ticket from the just created customer
        my $TicketID = $TicketObject->TicketCreate(
            Title        => 'Selenium Test-Some Ticket Title',
            Queue        => 'Raw',
            Lock         => 'unlock',
            Priority     => '3 normal',
            State        => 'closed successful',
            CustomerUser => $TestUserCustomer,
            OwnerID      => 1,
            UserID       => 1,
        );

        $Self->True(
            $TicketID,
            "Ticket is created - $TicketID",
        );

        my $TestService = "Service-" . $HelperObject->GetRandomID();

        # create a test service
        my %ServiceValues = (
            Name    => $TestService,
            Comment => 'Selenium Test Service',
            ValidID => 1,
            UserID  => 1,
        );

        if ($IsITSMIncidentProblemManagementInstalled) {
            $ServiceValues{TypeID}      = 1;
            $ServiceValues{Criticality} = '3 normal';
        }

        my $ServiceID = $ServiceObject->ServiceAdd(
            %ServiceValues
        );
        $Self->True(
            $ServiceID,
            "Service is created - $ServiceID",
        );

        # allow access to the just created service to the test user
        $ServiceObject->CustomerUserServiceMemberAdd(
            CustomerUserLogin => $TestUserCustomer,
            ServiceID         => $ServiceID,
            Active            => 1,
            UserID            => 1,
        );

        # create an article for the test ticket
        my $ArticleBackendObject = $ArticleObject->BackendForChannel(
            ChannelName => 'Internal',
        );
        my $ArticleID = $ArticleBackendObject->ArticleCreate(
            TicketID             => $TicketID,
            IsVisibleForCustomer => 0,
            SenderType           => 'agent',
            Subject              => 'Selenium test',
            Body                 => 'Just a test body for selenium testing',
            Charset              => 'ISO-8859-15',
            MimeType             => 'text/plain',
            HistoryType          => 'AddNote',
            HistoryComment       => 'Selenium testing',
            UserID               => 1,
        );

        $Self->True(
            $ArticleID,
            "Article is created - $ArticleID",
        );

        my $ScriptAlias = $ConfigObject->Get('ScriptAlias');

        # real selenium test start
        # open the page that clicking on Split link of the zoom view of the
        # just created ticket would open
        $Selenium->VerifiedGet(
            "${ScriptAlias}index.pl?Action=AgentTicketPhone;TicketID=$TicketID;ArticleID=$ArticleID"
        );

        # verify that the services dropdown has just created service
        $Self->True(
            $Selenium->find_element( "select#ServiceID option[value='$ServiceID']", 'css' ),
            "The services dropdown has created service - $TestService",
        );

        # get DB object
        my $DBObject = $Kernel::OM->Get('Kernel::System::DB');

        # clean up test data
        # delete the test ticket
        my $Success = $TicketObject->TicketDelete(
            TicketID => $TicketID,
            UserID   => 1,
        );

        # Ticket deletion could fail if apache still writes to ticket history. Try again in this case.
        if ( !$Success ) {
            sleep 3;
            $Success = $TicketObject->TicketDelete(
                TicketID => $TicketID,
                UserID   => 1,
            );
        }

        $Self->True(
            $Success,
            "Deleted ticket - $TicketID",
        );

        # delete the test service
        $Success = $DBObject->Do(
            SQL => "DELETE FROM service_customer_user WHERE service_id = $ServiceID",
        );
        $Self->True(
            $Success,
            "ServiceCustomerUser deleted - $ServiceID",
        );

        $Success = $DBObject->Do(
            SQL => "DELETE FROM service WHERE id = $ServiceID",
        );
        $Self->True(
            $Success,
            "Deleted Service - $ServiceID",
        );

        # make sure the cache is correct.
        for my $Cache (
            qw (Service Ticket)
            )
        {
            $CacheObject->CleanUp(
                Type => $Cache,
            );
        }
    }
);

1;
